<?php

class AdminController extends Controller
{
  public function __construct()
  {
    require_once APP_PATH . '/core/Helpers.php';

    $logFile = dirname(__DIR__, 2) . '/debug_admin.txt';


    if (!isset($_SESSION['user_role']) || $_SESSION['user_role'] !== 'admin') {
      file_put_contents($logFile, "Access Denied. Redirecting to login.\n", FILE_APPEND);
      $this->redirect('/login');
    }

    // Ensure expired orders are cancelled when Admin views them
    $orderModel = $this->model('Order');
    $orderModel->cancelExpiredPending(10);
  }

  public function dashboard()
  {
    $filters = [];
    if (!empty($_GET['event_id']))
      $filters['event_id'] = $_GET['event_id'];
    if (!empty($_GET['ticket_id']))
      $filters['ticket_id'] = $_GET['ticket_id'];
    if (!empty($_GET['start_date']))
      $filters['start_date'] = $_GET['start_date'];
    if (!empty($_GET['end_date']))
      $filters['end_date'] = $_GET['end_date'];

    $userModel = $this->model('User');
    $eventModel = $this->model('Event');
    $orderModel = $this->model('Order');
    $ticketModel = $this->model('TicketType');

    // Fetch Stats with Filters
    $stats = [
      'users' => $userModel->count(), // Users global
      'events' => $eventModel->count(), // Events global
      'orders' => $orderModel->count($filters),
      'revenue' => $orderModel->sumTotal($filters)
    ];

    // Fetch Lists for Filters
    $events = $eventModel->getAll();
    $tickets = [];
    if (!empty($filters['event_id'])) {
      $tickets = $ticketModel->getByEventId($filters['event_id']);
    }

    // Charts Data
    $dailySales = $orderModel->getDailySales($filters);

    $salesByTicket = [];
    if (!empty($filters['event_id'])) {
      $salesByTicket = $orderModel->getSalesByTicketType($filters);
    } else {
      // If no event selected, maybe show Top Events?
      $salesByTicket = $orderModel->getSalesByEvent($filters); // Reuse format: title/name, revenue
    }

    // Check-in Stats
    $userTicketModel = $this->model('UserTicket');
    $checkinStats = $userTicketModel->getCheckinStats(5, $filters);

    // Recent Orders
    $recentOrders = $orderModel->getRecentOrders(10, $filters);

    $this->view('admin/dashboard', [
      'stats' => $stats,
      'events' => $events,
      'tickets' => $tickets,
      'filters' => $filters,
      'dailySales' => $dailySales,
      'salesByData' => $salesByTicket,
      'recentOrders' => $recentOrders
    ]);
  }

  public function clients()
  {
    $userModel = $this->model('User');
    $clients = $userModel->getAll();
    $this->view('admin/clients', ['clients' => $clients]);
  }

  public function events()
  {
    $eventModel = $this->model('Event');
    $events = $eventModel->getAll();
    $this->view('admin/events', ['events' => $events]);
  }

  public function createEvent()
  {
    $this->view('admin/create_event');
  }

  public function storeEvent()
  {
    require_once APP_PATH . '/core/Csrf.php';
    if (!Csrf::validate($_POST['csrf_token'] ?? '')) {
      die('CSRF Validation Failed');
    }


    $data = [
      'title' => filter_input(INPUT_POST, 'title', FILTER_SANITIZE_SPECIAL_CHARS),
      'slug' => filter_input(INPUT_POST, 'slug', FILTER_SANITIZE_SPECIAL_CHARS),
      'description' => $_POST['description'],
      'location' => filter_input(INPUT_POST, 'location', FILTER_SANITIZE_SPECIAL_CHARS),
      'start_date' => $_POST['start_date'],
      'status' => $_POST['status'],
      'organizer_name' => filter_input(INPUT_POST, 'organizer_name', FILTER_SANITIZE_SPECIAL_CHARS),
      'organizer_phone' => filter_input(INPUT_POST, 'organizer_phone', FILTER_SANITIZE_SPECIAL_CHARS),
      'map_embed' => Helpers::extractMapSrc($_POST['map_embed'] ?? ''),
      'marketing_content' => $_POST['marketing_content'],
      'social_instagram' => filter_input(INPUT_POST, 'social_instagram', FILTER_SANITIZE_URL),
      'social_linkedin' => filter_input(INPUT_POST, 'social_linkedin', FILTER_SANITIZE_URL),
      'social_website' => filter_input(INPUT_POST, 'social_website', FILTER_SANITIZE_URL),
      'social_whatsapp' => filter_input(INPUT_POST, 'social_whatsapp', FILTER_SANITIZE_SPECIAL_CHARS)
    ];

    require_once APP_PATH . '/core/Helpers.php';
    $data['image_cover'] = '';
    if (isset($_FILES['image_cover']) && $_FILES['image_cover']['error'] === UPLOAD_ERR_OK) {
      $data['image_cover'] = Helpers::uploadFile($_FILES['image_cover'], 'events');
    }

    $eventModel = $this->model('Event');
    $eventId = $eventModel->create($data); // Now expects ID
    if ($eventId) {
      // Save Speakers
      if (!empty($_POST['speakers']) && is_array($_POST['speakers'])) {
        $speakerModel = $this->model('EventSpeaker');
        foreach ($_POST['speakers'] as $key => $speaker) {
          if (!empty($speaker['name']) && !empty($speaker['topic'])) {

            $imageUrl = '';
            // Handle File Upload for this index/key
            if (isset($_FILES['speakers']['name'][$key]['image_file'])) {
              $fileData = [
                'name' => $_FILES['speakers']['name'][$key]['image_file'],
                'type' => $_FILES['speakers']['type'][$key]['image_file'],
                'tmp_name' => $_FILES['speakers']['tmp_name'][$key]['image_file'],
                'error' => $_FILES['speakers']['error'][$key]['image_file'],
                'size' => $_FILES['speakers']['size'][$key]['image_file']
              ];
              if ($fileData['error'] === UPLOAD_ERR_OK) {
                $imageUrl = Helpers::uploadFile($fileData, 'speakers');
              }
            }

            $speakerModel->create([
              'event_id' => $eventId,
              'name' => htmlspecialchars($speaker['name']),
              'topic' => htmlspecialchars($speaker['topic']),
              'image_url' => $imageUrl
            ]);
          }
        }
      }
      $this->redirect('/admin/dashboard?msg=evento_criado');
    } else {
      die('Erro ao criar evento');
    }
  }

  public function manageTickets()
  {
    $eventId = $_GET['id'] ?? null;
    if (!$eventId) {
      $this->redirect('/admin/events');
    }

    $eventModel = $this->model('Event');
    $event = $eventModel->findById($eventId);

    if (!$event) {
      die('Evento não encontrado');
    }

    $ticketModel = $this->model('TicketType');
    $ticketTypes = $ticketModel->getByEventId($eventId);

    $this->view('admin/manage_tickets', ['event' => $event, 'ticketTypes' => $ticketTypes]);
  }

  public function storeTicket()
  {
    require_once APP_PATH . '/core/Csrf.php';
    if (!Csrf::validate($_POST['csrf_token'] ?? '')) {
      die('CSRF Validation Failed');
    }

    $data = [
      'event_id' => $_POST['event_id'],
      'name' => filter_input(INPUT_POST, 'name', FILTER_SANITIZE_SPECIAL_CHARS),
      'batch_name' => filter_input(INPUT_POST, 'batch_name', FILTER_SANITIZE_SPECIAL_CHARS),
      'price' => $_POST['price'],
      'promotional_price' => $_POST['promotional_price'],
      'total_quantity' => $_POST['total_quantity'],
      'promotion_start_date' => $_POST['promotion_start_date'],
      'promotion_end_date' => $_POST['promotion_end_date']
    ];

    $ticketModel = $this->model('TicketType');
    if ($ticketModel->create($data)) {
      $this->redirect('/admin/events/tickets?id=' . $data['event_id']);
    } else {
      die('Erro ao criar ingresso');
    }
  }

  public function editTicket()
  {
    $id = $_GET['id'] ?? null;
    if (!$id)
      $this->redirect('/admin/events');

    $ticketModel = $this->model('TicketType');
    $ticket = $ticketModel->findById($id);

    if (!$ticket)
      $this->redirect('/admin/events');

    $this->view('admin/edit_ticket', ['ticket' => $ticket]);
  }

  public function updateTicket()
  {
    require_once APP_PATH . '/core/Csrf.php';
    if (!Csrf::validate($_POST['csrf_token'] ?? '')) {
      die('CSRF Validation Failed');
    }

    $id = $_POST['id'];
    $data = [
      'name' => filter_input(INPUT_POST, 'name', FILTER_SANITIZE_SPECIAL_CHARS),
      'batch_name' => filter_input(INPUT_POST, 'batch_name', FILTER_SANITIZE_SPECIAL_CHARS),
      'price' => $_POST['price'],
      'promotional_price' => $_POST['promotional_price'],
      'total_quantity' => $_POST['total_quantity'],
      'promotion_start_date' => $_POST['promotion_start_date'],
      'promotion_end_date' => $_POST['promotion_end_date'],
      'is_active' => $_POST['is_active']
    ];

    $ticketModel = $this->model('TicketType');
    if ($ticketModel->update($id, $data)) {
      $ticket = $ticketModel->findById($id);
      $this->redirect('/admin/events/tickets?id=' . $ticket['event_id'] . '&msg=ingresso_atualizado');
    } else {
      die("Erro ao atualizar ingresso");
    }
  }

  public function deleteTicket()
  {
    $id = $_GET['id'] ?? null;

    if (!$id)
      $this->redirect('/admin/events');

    $ticketModel = $this->model('TicketType');
    $ticket = $ticketModel->findById($id);

    if ($ticket) {
      // Check for sales
      $userTicketModel = $this->model('UserTicket');
      $salesCount = $userTicketModel->countByTicketType($id);

      if ($salesCount > 0) {
        // Has sales, cannot delete. Deactivate instead?
        // For now, just error message.
        $this->redirect('/admin/events/tickets?id=' . $ticket['event_id'] . '&msg=erro_ingresso_com_vendas');
      } else {
        $ticketModel->delete($id);
        $this->redirect('/admin/events/tickets?id=' . $ticket['event_id'] . '&msg=ingresso_excluido');
      }
    } else {
      $this->redirect('/admin/events');
    }
  }

  public function checkin()
  {
    $this->view('admin/checkin');
  }

  public function processCheckin()
  {
    require_once APP_PATH . '/core/Csrf.php';
    if (!Csrf::validate($_POST['csrf_token'] ?? '')) {
      die('CSRF Validation Failed');
    }

    $code = $_POST['code'] ?? '';
    $userTicketModel = $this->model('UserTicket');

    $ticket = $userTicketModel->findByCode($code);

    if ($ticket) {
      if ($ticket['status'] === 'used') {
        $this->view('admin/checkin', ['error' => 'Ingresso JÁ UTILIZADO em: ' . date('d/m/Y H:i', strtotime($ticket['used_at']))]);
      } elseif ($ticket['status'] === 'cancelled') {
        $this->view('admin/checkin', ['error' => 'Ingresso CANCELADO. Acesso negado.']);
      } elseif ($ticket['status'] === 'pending' || $ticket['status'] === 'pending_payment') {
        $this->view('admin/checkin', ['error' => 'Pagamento PENDENTE. Não liberar entrada.']);
      } elseif ($ticket['status'] === 'active') {
        $userTicketModel->markAsUsed($ticket['id']);
        $this->view('admin/checkin', ['success' => true, 'ticket' => $ticket]);
      } else {
        $this->view('admin/checkin', ['error' => 'Status desconhecido (' . $ticket['status'] . '). Consulte suporte.']);
      }
    } else {
      $this->view('admin/checkin', ['error' => 'Código inválido ou não encontrado.']);
    }
  }

  public function editEvent()
  {
    $id = $_GET['id'] ?? null;
    if (!$id)
      $this->redirect('/admin/events');

    $eventModel = $this->model('Event');
    $event = $eventModel->findById($id);

    if (!$event)
      $this->redirect('/admin/events');

    $eventSpeakerModel = $this->model('EventSpeaker');
    $speakers = $eventSpeakerModel->getByEventId($id);

    $this->view('admin/edit_event', ['event' => $event, 'speakers' => $speakers]);
  }

  public function updateEvent()
  {
    require_once APP_PATH . '/core/Csrf.php';
    if (!Csrf::validate($_POST['csrf_token'] ?? '')) {
      die('CSRF Validation Failed');
    }

    $id = $_POST['id'];

    $data = [
      'title' => filter_input(INPUT_POST, 'title', FILTER_SANITIZE_SPECIAL_CHARS),
      'slug' => filter_input(INPUT_POST, 'slug', FILTER_SANITIZE_SPECIAL_CHARS),
      'description' => $_POST['description'],
      'location' => filter_input(INPUT_POST, 'location', FILTER_SANITIZE_SPECIAL_CHARS),
      'start_date' => $_POST['start_date'],
      'status' => $_POST['status'],
      'organizer_name' => filter_input(INPUT_POST, 'organizer_name', FILTER_SANITIZE_SPECIAL_CHARS),
      'organizer_phone' => filter_input(INPUT_POST, 'organizer_phone', FILTER_SANITIZE_SPECIAL_CHARS),
      'map_embed' => Helpers::extractMapSrc($_POST['map_embed'] ?? ''),
      'marketing_content' => $_POST['marketing_content'] ?? null,
      'social_instagram' => filter_input(INPUT_POST, 'social_instagram', FILTER_SANITIZE_URL),
      'social_linkedin' => filter_input(INPUT_POST, 'social_linkedin', FILTER_SANITIZE_URL),
      'social_website' => filter_input(INPUT_POST, 'social_website', FILTER_SANITIZE_URL),
      'social_whatsapp' => filter_input(INPUT_POST, 'social_whatsapp', FILTER_SANITIZE_SPECIAL_CHARS)
    ];

    require_once APP_PATH . '/core/Helpers.php';
    if (isset($_FILES['image_cover']) && $_FILES['image_cover']['error'] === UPLOAD_ERR_OK) {
      $data['image_cover'] = Helpers::uploadFile($_FILES['image_cover'], 'events');
    } else {
      $data['image_cover'] = $_POST['current_image_cover'];
    }

    $eventModel = $this->model('Event');
    if ($eventModel->update($id, $data)) {
      // Sync Speakers
      $speakerModel = $this->model('EventSpeaker');
      $speakerModel->deleteByEventId($id);

      if (!empty($_POST['speakers']) && is_array($_POST['speakers'])) {
        foreach ($_POST['speakers'] as $key => $speaker) {
          if (!empty($speaker['name']) && !empty($speaker['topic'])) {

            $imageUrl = $speaker['current_image'] ?? '';
            // Handle File Upload for this index/key
            if (isset($_FILES['speakers']['name'][$key]['image_file'])) {
              $fileData = [
                'name' => $_FILES['speakers']['name'][$key]['image_file'],
                'type' => $_FILES['speakers']['type'][$key]['image_file'],
                'tmp_name' => $_FILES['speakers']['tmp_name'][$key]['image_file'],
                'error' => $_FILES['speakers']['error'][$key]['image_file'],
                'size' => $_FILES['speakers']['size'][$key]['image_file']
              ];
              if ($fileData['error'] === UPLOAD_ERR_OK) {
                $uploaded = Helpers::uploadFile($fileData, 'speakers');
                if ($uploaded) {
                  $imageUrl = $uploaded;
                }
              }
            }

            $speakerModel->create([
              'event_id' => $id,
              'name' => htmlspecialchars($speaker['name']),
              'topic' => htmlspecialchars($speaker['topic']),
              'image_url' => $imageUrl
            ]);
          }
        }
      }

      $this->redirect('/admin/events?msg=evento_atualizado');
    } else {
      die("Erro ao atualizar evento");
    }
  }

  public function deleteEvent()
  {
    $id = $_GET['id'] ?? null;
    if (!$id)
      $this->redirect('/admin/events');

    try {
      // 1. Check for sales first (UserTickets)
      $userTicketModel = $this->model('UserTicket');
      $attendees = $userTicketModel->getAttendeesByEvent($id);

      if (count($attendees) > 0) {
        $this->redirect('/admin/events?msg=erro_evento_com_vendas');
        return;
      }

      // 2. Delete Speakers
      $speakerModel = $this->model('EventSpeaker');
      $speakerModel->deleteByEventId($id);

      // 3. Delete Tickets
      $ticketModel = $this->model('TicketType');
      $tickets = $ticketModel->getByEventId($id);
      foreach ($tickets as $t) {
        $ticketModel->delete($t['id']);
      }

      // 4. Delete Event
      $eventModel = $this->model('Event');
      if ($eventModel->delete($id)) {
        $this->redirect('/admin/events?msg=evento_excluido');
      } else {
        throw new Exception("Falha ao deletar registro de evento.");
      }
    } catch (Exception $e) {
      $this->redirect('/admin/events?msg=erro_ao_excluir');
    }
  }

  public function setHighlight()
  {
    $id = $_GET['id'] ?? null;
    if (!$id)
      $this->redirect('/admin/events');

    $eventModel = $this->model('Event');

    // Clear all highlights
    $eventModel->clearHighlights();

    // Set new highlight
    $eventModel->markAsHighlight($id);

    $this->redirect('/admin/events?msg=evento_destaque_definido');
  }

  public function reports()
  {
    $eventId = $_GET['event_id'] ?? null;
    $reportType = $_GET['type'] ?? 'sales'; // sales, attendees
    $ticketId = $_GET['ticket_id'] ?? null;

    $eventModel = $this->model('Event');
    $orderModel = $this->model('Order');
    $userTicketModel = $this->model('UserTicket');
    $ticketModel = $this->model('TicketType');

    $events = $eventModel->getAll();
    $tickets = [];
    $reportData = [];

    // Fetch tickets if event selected
    if ($eventId) {
      $tickets = $ticketModel->getByEventId($eventId);
    }

    $filters = ['event_id' => $eventId, 'ticket_id' => $ticketId, 'type' => $reportType];

    if ($reportType === 'sales') {
      if ($eventId) {
        // Specific event sales (with optional ticket filter handled by model)
        $reportData = $orderModel->getSalesByTicketType($filters);
      } else {
        // Global event sales
        $reportData = $orderModel->getSalesByEvent();
      }
    } elseif ($reportType === 'attendees') {
      if ($eventId) {
        $reportData = $userTicketModel->getAttendeesByEvent($eventId, $filters);
      } else {
        // Requiring event ID for attendees list for now
        $reportData = [];
      }
    }

    $this->view('admin/reports', [
      'events' => $events,
      'tickets' => $tickets,
      'reportData' => $reportData,
      'filters' => $filters
    ]);
  }

  public function exportReport()
  {
    $eventId = $_GET['event_id'] ?? null;
    $reportType = $_GET['type'] ?? 'sales';
    $ticketId = $_GET['ticket_id'] ?? null;
    $format = $_GET['format'] ?? 'csv';

    $orderModel = $this->model('Order');
    $userTicketModel = $this->model('UserTicket');
    $eventModel = $this->model('Event'); // To get event name

    $data = [];
    $eventName = null;

    if ($eventId) {
      $event = $eventModel->findById($eventId);
      $eventName = $event ? $event['title'] : null;
    }

    if ($reportType === 'sales') {
      if ($eventId) {
        $data = $orderModel->getSalesByTicketType(['event_id' => $eventId, 'ticket_id' => $ticketId]);
      } else {
        $data = $orderModel->getSalesByEvent();
      }
    } elseif ($reportType === 'attendees' && $eventId) {
      $data = $userTicketModel->getAttendeesByEvent($eventId, ['ticket_id' => $ticketId]);
    }

    // PDF (Print View)
    if ($format === 'pdf') {
      $this->view('admin/reports_print', [
        'reportData' => $data,
        'type' => $reportType,
        'eventName' => $eventName,
        'filters' => ['event_id' => $eventId]
      ]);
      return;
    }

    // CSV Export
    $filename = "report_{$reportType}_" . date('Y-m-d_His') . ".csv";

    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="' . $filename . '"');

    $output = fopen('php://output', 'w');

    if ($reportType === 'sales') {
      if ($eventId) {
        fputcsv($output, ['Ingresso', 'Vendas', 'Receita']);
        foreach ($data as $row) {
          fputcsv($output, [$row['name'], $row['count'], $row['revenue'] ?? 0]);
        }
      } else {
        fputcsv($output, ['ID', 'Evento', 'Pedidos', 'Receita']);
        foreach ($data as $row) {
          fputcsv($output, [$row['id'], $row['title'], $row['total_orders'], $row['total_revenue'] ?? 0]);
        }
      }
    } elseif ($reportType === 'attendees' && $eventId) {
      fputcsv($output, ['Nome', 'Email', 'Ingresso', 'Código', 'Data Compra', 'Status', 'Check-in']);
      foreach ($data as $row) {
        fputcsv($output, [
          $row['user_name'],
          $row['user_email'],
          $row['ticket_name'],
          $row['unique_code'],
          $row['purchase_date'],
          $row['status'],
          $row['used_at'] ?? 'N/A'
        ]);
      }
    }

    fclose($output);
    exit;
  }

  public function settings()
  {
    $settingModel = $this->model('Setting');
    $settings = $settingModel->getAllAsArray();
    $this->view('admin/settings', ['settings' => $settings]);
  }

  public function updateSettings()
  {
    require_once APP_PATH . '/core/Csrf.php';
    if (!Csrf::validate($_POST['csrf_token'] ?? '')) {
      $this->redirect('/admin/settings?error=csrf');
      return;
    }

    // Filter allowed keys to prevent pollution if we want strictness, 
    // but for admin panel flexibility we'll trust the form fields map to known keys
    // or just save everything posted that matches our known keys.
    $allowedKeys = [
      'site_name',
      'contact_email',
      'contact_whatsapp',
      'maintenance_mode',
      'asaas_api_key',
      'asaas_url',
      'asaas_webhook_token',
      'smtp_host',
      'smtp_port',
      'smtp_user',
      'smtp_pass',
      'color_brand_neon',
      'color_brand_secondary',
      'color_bg_primary',
      'seo_title_suffix',
      'seo_description_default',
      'seo_keywords_default',
      'agenda_hero_title',
      'agenda_hero_subtitle',
      'agenda_seo_title',
      'agenda_seo_description',
      'site_logo',
      'site_logo_inverted',
      'site_favicon',
      'email_logo',
      'email_verification_enabled'
    ];

    $data = [];

    // Handle Uploads
    require_once APP_PATH . '/core/Helpers.php';

    // Logo (Dark Mode / Default)
    if (isset($_FILES['site_logo']) && $_FILES['site_logo']['error'] === UPLOAD_ERR_OK) {
      $logoPath = Helpers::uploadFile($_FILES['site_logo'], 'settings');
      if ($logoPath) {
        $data['site_logo'] = $logoPath;
      }
    }

    // Logo (Light Mode)
    if (isset($_FILES['site_logo_inverted']) && $_FILES['site_logo_inverted']['error'] === UPLOAD_ERR_OK) {
      $logoPathInv = Helpers::uploadFile($_FILES['site_logo_inverted'], 'settings');
      if ($logoPathInv) {
        $data['site_logo_inverted'] = $logoPathInv;
      }
    }

    // Favicon
    if (isset($_FILES['site_favicon']) && $_FILES['site_favicon']['error'] === UPLOAD_ERR_OK) {
      $faviconPath = Helpers::uploadFile($_FILES['site_favicon'], 'settings');
      if ($faviconPath) {
        $data['site_favicon'] = $faviconPath;
      }
    }

    // Email Logo
    if (isset($_FILES['email_logo']) && $_FILES['email_logo']['error'] === UPLOAD_ERR_OK) {
      $emailLogoPath = Helpers::uploadFile($_FILES['email_logo'], 'settings');
      if ($emailLogoPath) {
        $data['email_logo'] = $emailLogoPath;
      }
    }

    // Process other fields (without resetting data)
    foreach ($allowedKeys as $key) {
      if (isset($_POST[$key])) {
        $data[$key] = $_POST[$key];
      } else {
        // Checkbox handling
        if ($key === 'maintenance_mode' || $key === 'email_verification_enabled')
          $data[$key] = '0';
      }
    }

    $settingModel = $this->model('Setting');
    $settingModel->updateBatch($data);

    // Persist to .env
    $envMap = [
      'asaas_api_key' => 'ASAAS_API_KEY',
      'asaas_url' => 'ASAAS_URL',
      'asaas_webhook_token' => 'ASAAS_WEBHOOK_TOKEN',
      'smtp_host' => 'SMTP_HOST',
      'smtp_port' => 'SMTP_PORT',
      'smtp_user' => 'SMTP_USER',
      'smtp_pass' => 'SMTP_PASS'
    ];

    foreach ($envMap as $dbKey => $envKey) {
      if (isset($data[$dbKey])) {
        Helpers::updateEnv($envKey, $data[$dbKey]);
      }
    }

    $this->redirect('/admin/settings?msg=configuracoes_salvas');
  }

  public function testEmail()
  {
    $testEmail = $_POST['test_email'] ?? '';
    if (!filter_var($testEmail, FILTER_VALIDATE_EMAIL)) {
      $this->redirect('/admin/settings?msg=email_invalido');
    }

    require_once APP_PATH . '/core/Helpers.php';
    $sent = Helpers::sendEmail($testEmail, 'Teste de SMTP - Cê Vai', '<h1>Teste de SMTP</h1><p>Se você recebeu este e-mail, a configuração de SMTP está funcionando!</p>');

    if ($sent) {
      $this->redirect('/admin/settings?msg=email_teste_enviado');
    } else {
      $this->redirect('/admin/settings?msg=erro_envio_email');
    }
  }

  // Speaker Management
  public function manageSpeakers()
  {
    $eventId = $_GET['id'] ?? null;
    if (!$eventId)
      $this->redirect('/admin/events');

    $eventModel = $this->model('Event');
    $event = $eventModel->findById($eventId);

    $speakerModel = $this->model('Speaker');
    $speakers = $speakerModel->getByEventId($eventId);

    $this->view('admin/manage_speakers', ['event' => $event, 'speakers' => $speakers]);
  }

  public function storeSpeaker()
  {
    require_once APP_PATH . '/core/Csrf.php';
    if (!Csrf::validate($_POST['csrf_token'] ?? '')) {
      die('CSRF Validation Failed');
    }

    $data = [
      'event_id' => $_POST['event_id'],
      'name' => filter_input(INPUT_POST, 'name', FILTER_SANITIZE_SPECIAL_CHARS),
      'role' => filter_input(INPUT_POST, 'role', FILTER_SANITIZE_SPECIAL_CHARS),
      'image_url' => filter_input(INPUT_POST, 'image_url', FILTER_SANITIZE_URL),
      'bio' => $_POST['bio']
    ];

    $speakerModel = $this->model('Speaker');
    if ($speakerModel->create($data)) {
      $this->redirect('/admin/events/speakers?id=' . $data['event_id']);
    } else {
      die("Erro ao adicionar palestrante");
    }
  }

  public function deleteSpeaker()
  {
    $id = $_GET['id']; // Warning: undefined index if missing
    $eventId = $_GET['event_id'] ?? null;

    if (!$id || !$eventId) {
      $this->redirect('/admin/events');
      return;
    }

    try {
      $speakerModel = $this->model('EventSpeaker');
      if ($speakerModel->delete($id)) {
        $this->redirect('/admin/events/speakers?id=' . $eventId . '&msg=palestrante_excluido');
      } else {
        throw new Exception("Falha DB ao excluir palestrante");
      }
    } catch (Exception $e) {
      file_put_contents('debug_error.log', "[" . date('Y-m-d H:i:s') . "] Delete Speaker Error: " . $e->getMessage() . "\n", FILE_APPEND);
      $this->redirect('/admin/events/speakers?id=' . $eventId . '&msg=erro_ao_excluir');
    }
  }

  // Client Management
  public function editClient()
  {
    $id = $_GET['id'] ?? null;
    if (!$id)
      $this->redirect('/admin/clients');

    $userModel = $this->model('User');
    $client = $userModel->findById($id);

    if (!$client)
      $this->redirect('/admin/clients');

    $this->view('admin/edit_client', ['client' => $client]);
  }

  public function updateClient()
  {
    $id = $_POST['id'];
    $data = [
      'name' => filter_input(INPUT_POST, 'name', FILTER_SANITIZE_SPECIAL_CHARS),
      'phone' => filter_input(INPUT_POST, 'phone', FILTER_SANITIZE_SPECIAL_CHARS)
    ];

    // Password update logic if provided
    if (!empty($_POST['password'])) {
      $data['password'] = $_POST['password'];
    }

    $userModel = $this->model('User');
    if ($userModel->update($id, $data)) {
      $this->redirect('/admin/clients?msg=cliente_atualizado');
    } else {
      die("Erro ao atualizar cliente");
    }
  }

  public function deleteClient()
  {
    $id = $_GET['id'] ?? null;
    if (!$id)
      $this->redirect('/admin/clients');

    try {
      // Check for orders
      $orderModel = $this->model('Order');
      $orderCount = $orderModel->countByUserId($id);

      if ($orderCount > 0) {
        $this->redirect('/admin/clients?msg=erro_cliente_com_pedidos');
        return;
      }

      $userModel = $this->model('User');
      if ($userModel->delete($id)) {
        $this->redirect('/admin/clients?msg=cliente_excluido');
      } else {
        throw new Exception("Falha DB ao excluir cliente");
      }
    } catch (Exception $e) {
      file_put_contents('debug_error.log', "[" . date('Y-m-d H:i:s') . "] Delete Client Error: " . $e->getMessage() . "\n", FILE_APPEND);
      $this->redirect('/admin/clients?msg=erro_ao_excluir');
    }
  }

  // Order Management
  public function orders()
  {
    $orderModel = $this->model('Order');
    // Assuming getRecentOrders can be used without limit or we add a getAll method
    // For now, let's fetch recent 50 or use getAll if exists. 
    // Checking Order model, it has getRecentOrders($limit, $filters).
    $orders = $orderModel->getRecentOrders(50);

    $this->view('admin/orders', ['orders' => $orders]);
  }

  public function order()
  {
    $id = $_GET['id'] ?? null;
    if (!$id)
      $this->redirect('/admin/dashboard');

    $orderModel = $this->model('Order');
    $order = $orderModel->findById($id);

    if (!$order)
      die("Pedido não encontrado");

    $userModel = $this->model('User');
    $user = $userModel->findById($order['user_id']);

    // Fetch items/tickets logic would ideally be in Order or Ticket model to get by order_id
    // For now mocking or fetching directly if method exists or adding custom sql here/model
    // Adding getTicketsByOrderId to Order Model on the fly or using existing relationship

    // We need a method to get tickets for an order
    // Let's assume we implement getItemsByOrder in Order model next
    $items = $orderModel->getItemsByOrder($id);

    $this->view('admin/order_details', ['order' => $order, 'user' => $user, 'items' => $items]);
  }
  // --- Order Management Actions ---

  public function editOrder()
  {
    $id = $_GET['id'] ?? null;
    if (!$id)
      $this->redirect('/admin/orders');

    $orderModel = $this->model('Order');
    $order = $orderModel->findById($id);

    if (!$order)
      die("Pedido não encontrado");

    $userModel = $this->model('User');
    $user = $userModel->findById($order['user_id']);

    $this->view('admin/edit_order', ['order' => $order, 'user' => $user]);
  }

  public function updateOrder()
  {
    $id = $_POST['id'];
    $status = $_POST['payment_status'];

    $orderModel = $this->model('Order');
    $orderModel->updateStatus($id, $status);

    // Sync Ticket Status
    $userTicketModel = $this->model('UserTicket');

    // Determine ticket status based on order status
    $ticketStatus = 'pending'; // default
    if ($status === 'paid') {
      $ticketStatus = 'active';
    } elseif ($status === 'cancelled') {
      $ticketStatus = 'cancelled';
    }

    $userTicketModel->updateStatusByOrder($id, $ticketStatus);

    $this->redirect('/admin/orders?msg=pedido_atualizado');
  }

  public function resendOrderEmail()
  {
    $orderId = $_POST['order_id'];

    $orderModel = $this->model('Order');
    $order = $orderModel->findById($orderId);
    if (!$order)
      die("Pedido invalido");

    $userModel = $this->model('User');
    $user = $userModel->findById($order['user_id']);

    // Use getItemsByOrder which we confirmed returns ticket details including unique_code
    $items = $orderModel->getItemsByOrder($orderId);

    require_once APP_PATH . '/services/EmailService.php';
    $emailService = new EmailService();

    // Fix: We need full event data for the email
    $ticketModel = $this->model('TicketType');
    $eventModel = $this->model('Event');

    $cnt = 0;
    foreach ($items as $item) {
      // Fetch Ticket & Event to get full details (Location, Date, etc)
      $ticket = $ticketModel->findById($item['ticket_type_id']);
      if ($ticket) {
        $eventData = $eventModel->findById($ticket['event_id']);
        // Check if eventData is valid
        if (!$eventData)
          $eventData = ['title' => $item['event_title'] ?? 'Evento'];

        $emailService->sendTicketPurchased($user['email'], $user['name'], $eventData, $item['unique_code']);
        $cnt++;
      }
    }
    $this->redirect('/admin/order?id=' . $orderId . '&msg=email_reenviado&count=' . $cnt);
  }

  public function createOrder()
  {
    $eventModel = $this->model('Event');
    $events = $eventModel->getAll();

    // Fetch all ticket types, organizing them by event in the view or here
    // Let's just pass all tickets and filter in JS
    $ticketModel = $this->model('TicketType');
    // We need a method to get ALL tickets from all events or loop events.
    // Assuming getByEventId is what we have.
    // Let's create an array of tickets keyed by event_id for easier JS handling
    $ticketsByEvent = [];
    foreach ($events as $event) {
      $ticketsByEvent[$event['id']] = $ticketModel->getByEventId($event['id']);
    }

    $this->view('admin/create_order', ['events' => $events, 'ticketsByEvent' => $ticketsByEvent]);
  }

  public function storeManualOrder()
  {
    $eventId = $_POST['event_id'];
    $ticketId = $_POST['ticket_id'];
    $quantity = (int) $_POST['quantity'];
    $paymentType = $_POST['payment_type']; // 'pix_manual' or 'courtesy'

    $clientName = filter_input(INPUT_POST, 'client_name', FILTER_SANITIZE_SPECIAL_CHARS);
    $clientEmail = filter_input(INPUT_POST, 'client_email', FILTER_SANITIZE_EMAIL);
    $clientPhone = filter_input(INPUT_POST, 'client_phone', FILTER_SANITIZE_SPECIAL_CHARS);
    $clientCpf = filter_input(INPUT_POST, 'client_cpf', FILTER_SANITIZE_SPECIAL_CHARS);

    if (!$eventId || !$ticketId || $quantity < 1 || !$clientEmail || !$clientName) {
      die("Dados inválidos. Preencha todos os campos obrigatórios.");
    }

    // 1. User Management
    $userModel = $this->model('User');
    $user = $userModel->findByEmail($clientEmail);
    $userId = null;

    if ($user) {
      $userId = $user['id'];
      // Optional: Update name/phone/cpf if provided and different? 
      // For now, respect existing data or assume admin knows current data.
    } else {
      // Create new user
      $randomPass = bin2hex(random_bytes(4)); // 8 chars random password
      $userId = $userModel->create($clientName, $clientEmail, $randomPass, $clientCpf, $clientPhone);
      if (!$userId)
        die("Erro ao criar usuário.");

      // Ideally send a "Welcome" email with the password.
    }

    // 2. Ticket & Price Logic
    $ticketModel = $this->model('TicketType');
    $ticket = $ticketModel->findById($ticketId);

    if (!$ticket)
      die("Ingresso inválido.");

    $unitPrice = ($paymentType === 'courtesy') ? 0.00 : $ticket['price'];
    $totalAmount = $unitPrice * $quantity;
    $paymentMethod = ($paymentType === 'courtesy') ? 'courtesy' : 'manual_pix';
    $status = 'paid'; // Manual orders are strictly for confirmed/paid or free stuff.

    // 3. Create Order
    $orderModel = $this->model('Order');
    $orderId = $orderModel->create($userId, $totalAmount, $paymentMethod, $status);

    if (!$orderId)
      die("Erro ao criar pedido.");

    // 4. Generate Tickets
    $userTicketModel = $this->model('UserTicket');
    require_once APP_PATH . '/services/EmailService.php';
    $emailService = new EmailService();
    $eventModel = $this->model('Event');
    $event = $eventModel->findById($eventId);

    for ($i = 0; $i < $quantity; $i++) {
      $uniqueCode = strtoupper(bin2hex(random_bytes(6)));
      $userTicketModel->create($orderId, $userId, $ticket['id'], $uniqueCode);

      // Send Email
      $emailService->sendTicketPurchased($clientEmail, $clientName, $event, $uniqueCode);
    }

    $this->redirect('/admin/order?id=' . $orderId . '&msg=venda_realizada_sucesso');
  }

  // --- Operators Management ---

  public function operators()
  {
    $userModel = $this->model('User');
    $operators = $userModel->getAllOperators();
    $this->view('admin/operators', ['operators' => $operators]);
  }

  public function storeOperator()
  {
    if (!Helpers::csrf_token())
      require_once APP_PATH . '/core/Csrf.php';
    if (!Csrf::validate($_POST['csrf_token'] ?? '')) {
      $this->redirect('/admin/operators?error=csrf');
      return;
    }

    $name = filter_input(INPUT_POST, 'name', FILTER_SANITIZE_SPECIAL_CHARS);
    $email = filter_input(INPUT_POST, 'email', FILTER_SANITIZE_EMAIL);
    $password = $_POST['password'];
    $phone = filter_input(INPUT_POST, 'phone', FILTER_SANITIZE_SPECIAL_CHARS);

    // Create logic
    $userModel = $this->model('User');
    if ($userModel->findByEmail($email)) {
      $this->redirect('/admin/operators?error=email_exists');
      return;
    }

    // Pass null for CPF as it is now nullable
    $id = $userModel->create($name, $email, $password, null, $phone);

    if ($id) {
      // Update to operator using Model method
      if ($userModel->updateRole($id, 'operator')) {
        $this->redirect('/admin/operators?msg=operador_criado');
      } else {
        // If role update fails (rare but possible), maybe delete user or log?
        // For now just success msg as user is created anyway, manual role fix possible by admin if editing user exist.
        // Or better error:
        $this->redirect('/admin/operators?msg=operador_criado_verifique_permissao');
      }
    } else {
      $this->redirect('/admin/operators?error=create_failed');
    }
  }

  public function deleteOperator()
  {
    $id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
    $userModel = $this->model('User');
    $userModel->delete($id);
    $this->redirect('/admin/operators?msg=operador_removido');
  }

  public function updateOperator()
  {
    if (!Helpers::csrf_token())
      require_once APP_PATH . '/core/Csrf.php';
    if (!Csrf::validate($_POST['csrf_token'] ?? '')) {
      $this->redirect('/admin/operators?error=csrf');
      return;
    }

    $id = filter_input(INPUT_POST, 'id', FILTER_VALIDATE_INT);
    $name = filter_input(INPUT_POST, 'name', FILTER_SANITIZE_SPECIAL_CHARS);
    $email = filter_input(INPUT_POST, 'email', FILTER_SANITIZE_EMAIL);
    $phone = filter_input(INPUT_POST, 'phone', FILTER_SANITIZE_SPECIAL_CHARS);
    $password = $_POST['password'] ?? '';

    $userModel = $this->model('User');

    // Check if email belongs to another user
    $existing = $userModel->findByEmail($email);
    if ($existing && $existing['id'] != $id) {
      $this->redirect('/admin/operators?error=email_exists');
      return;
    }

    $data = [
      'name' => $name,
      'email' => $email,
      'phone' => $phone
    ];

    if (!empty($password)) {
      $data['password'] = $password;
    }

    if ($userModel->update($id, $data)) {
      $this->redirect('/admin/operators?msg=operador_atualizado');
    } else {
      $this->redirect('/admin/operators?error=update_failed');
    }
  }

  public function manageEventOperators()
  {
    $eventId = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
    $eventModel = $this->model('Event');
    $userModel = $this->model('User');

    $event = $eventModel->findById($eventId);
    $currentOperators = $eventModel->getOperators($eventId);
    $allOperators = $userModel->getAllOperators();

    $this->view('admin/manage_event_operators', [
      'event' => $event,
      'current_operators' => $currentOperators,
      'all_operators' => $allOperators
    ]);
  }

  public function addEventOperator()
  {
    $log = dirname(__DIR__, 2) . '/debug_link_operator.txt';
    file_put_contents($log, "addEventOperator triggered\n", FILE_APPEND);

    if (!class_exists('Csrf')) {
      require_once APP_PATH . '/core/Csrf.php';
      file_put_contents($log, "Csrf class loaded manually\n", FILE_APPEND);
    }

    if (!Csrf::validate($_POST['csrf_token'] ?? '')) {
      file_put_contents($log, "CSRF Failed\n", FILE_APPEND);
      $this->redirect($_SERVER['HTTP_REFERER']);
      return;
    }

    $eventId = $_POST['event_id'];
    $userId = $_POST['user_id'];
    file_put_contents($log, "Linking Event $eventId to User $userId\n", FILE_APPEND);

    $eventModel = $this->model('Event');
    try {
      if ($eventModel->addOperator($eventId, $userId)) {
        file_put_contents($log, "Success\n", FILE_APPEND);
        $this->redirect("/admin/events/operators?id=$eventId&msg=adicionado");
      } else {
        file_put_contents($log, "Failed (Model returned false)\n", FILE_APPEND);
        // Should redirect with error
        $this->redirect("/admin/events/operators?id=$eventId&error=falha_ao_vincular");
      }
    } catch (Exception $e) {
      file_put_contents($log, "Exception: " . $e->getMessage() . "\n", FILE_APPEND);
      // Show error
      echo "Error: " . $e->getMessage();
      die();
    }
  }

  public function removeEventOperator()
  {
    $eventId = filter_input(INPUT_GET, 'event_id', FILTER_VALIDATE_INT);
    $userId = filter_input(INPUT_GET, 'user_id', FILTER_VALIDATE_INT);

    $eventModel = $this->model('Event');
    $eventModel->removeOperator($eventId, $userId);

    $this->redirect("/admin/events/operators?id=$eventId&msg=removido");
  }


}



