<?php

class AuthController extends Controller
{

  public function login()
  {
    $this->view('auth/login');
  }

  public function authenticate()
  {
    if (!Helpers::csrf_token()) {
      require_once APP_PATH . '/core/Csrf.php';
    }
    if (!Csrf::validate($_POST['csrf_token'] ?? '')) {
      $this->view('auth/login', ['error' => 'Sessão inválida (CSRF). Tente novamente.']);
      return;
    }

    $email = filter_input(INPUT_POST, 'email', FILTER_SANITIZE_EMAIL);
    $password = $_POST['password'] ?? '';

    $userModel = $this->model('User');
    $user = $userModel->findByEmail($email);

    if ($user) {
      if (password_verify($password, $user['password'])) {

        // Check if active
        if (isset($user['is_active']) && $user['is_active'] == 0) {
          $this->view('auth/login', ['error' => 'Conta pendente de ativação. Verifique seu e-mail.']);
          return;
        }

        $_SESSION['user_id'] = $user['id'];
        $_SESSION['user_name'] = $user['name'];
        $_SESSION['user_role'] = $user['role'];

        if ($user['role'] === 'admin') {
          $this->redirect('/admin/dashboard');
        } elseif ($user['role'] === 'operator') {
          $this->redirect('/operator/dashboard');
        } else {
          $this->redirect('/client/dashboard');
        }
      } else {
        $this->view('auth/login', ['error' => 'E-mail ou senha inválidos.']);
      }
    } else {
      $this->view('auth/login', ['error' => 'E-mail ou senha inválidos.']);
    }
  }

  public function activate($token)
  {
    $userModel = $this->model('User');
    if ($userModel->activateAccount($token)) {
      $this->view('auth/login', ['success' => 'Conta ativada com sucesso! Faça login.']);
    } else {
      $this->view('auth/login', ['error' => 'Token de ativação inválido ou expirado.']);
    }
  }

  public function register()
  {
    $this->view('auth/register');
  }

  public function store()
  {
    try {
      // Fix for "Class Csrf not found"
      require_once APP_PATH . '/core/Csrf.php';

      if (!Csrf::validate($_POST['csrf_token'] ?? '')) {
        $this->view('auth/register', ['error' => 'Sessão inválida (CSRF). Tente novamente.']);
        return;
      }

      $name = filter_input(INPUT_POST, 'name', FILTER_SANITIZE_SPECIAL_CHARS);
      $email = filter_input(INPUT_POST, 'email', FILTER_SANITIZE_EMAIL);
      $password = $_POST['password'];
      $cpf = filter_input(INPUT_POST, 'cpf', FILTER_SANITIZE_SPECIAL_CHARS);
      $phone = filter_input(INPUT_POST, 'phone', FILTER_SANITIZE_SPECIAL_CHARS);
      $terms = isset($_POST['terms']);

      if (!$terms) {
        $this->view('auth/register', ['error' => 'Você deve aceitar os Termos de Uso e Política de Privacidade.']);
        return;
      }

      $userModel = $this->model('User');

      // Check if exists
      if ($userModel->findByEmail($email)) {
        $this->view('auth/register', ['error' => 'E-mail já cadastrado.']);
        return;
      }

      // SIMPLIFIED LOGIC: Always Active, No Email Verification for now to fix 500
      $isActive = 1;
      $activationToken = null;

      // Attempt Create
      if ($userId = $userModel->create($name, $email, $password, $cpf, $phone, $isActive, $activationToken)) {
        // SUCCESS
        $_SESSION['user_id'] = $userId;
        $_SESSION['user_name'] = $name;
        $_SESSION['user_role'] = 'client';
        $this->redirect('/client/dashboard');
        return;
      } else {
        throw new Exception("Falha ao inserir no banco de dados.");
      }

    } catch (Throwable $e) {
      // Emergency Debug Output
      http_response_code(200);
      echo "<div style='background:red;color:white;padding:20px;font-family:sans-serif;'>";
      echo "<h1>CRITICAL REGISTRATION ERROR</h1>";
      echo "<h3>" . $e->getMessage() . "</h3>";
      echo "<pre>" . $e->getTraceAsString() . "</pre>";
      echo "</div>";
      die();
    }
  }

  public function logout()
  {
    session_destroy();
    $this->redirect('/');
  }

  // Password Recovery

  public function forgotPassword()
  {
    $this->view('auth/forgot_password');
  }

  public function sendResetLink()
  {
    $email = filter_input(INPUT_POST, 'email', FILTER_SANITIZE_EMAIL);
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
      $this->view('auth/forgot_password', ['error' => 'Email inválido.']);
      return;
    }

    $userModel = $this->model('User');
    $user = $userModel->findByEmail($email);

    if (!$user) {
      // Security: Don't reveal user doesn't exist
      $this->view('auth/forgot_password', ['success' => 'Se o email existir, um link de recuperação foi enviado.']);
      return;
    }

    // Generate Token
    $token = bin2hex(random_bytes(32));

    // Store in DB
    $db = new Database();
    $conn = $db->connect();

    // Cleanup old tokens
    $conn->prepare("DELETE FROM password_resets WHERE email = ?")->execute([$email]);

    // Insert new
    $stmt = $conn->prepare("INSERT INTO password_resets (email, token, created_at) VALUES (?, ?, NOW())");
    $stmt->execute([$email, $token]);

    // Send Email
    require_once APP_PATH . '/core/Helpers.php';
    $link = APP_URL . "/reset-password/" . $token;
    $subject = "Recuperação de Senha - Cê Vai";
    $body = "<h2>Recuperação de Senha</h2>
             <p>Você solicitou a redefinição de sua senha.</p>
             <p>Clique no link abaixo para criar uma nova senha:</p>
             <p><a href='$link'>$link</a></p>
             <p>Se não foi você, ignore este email.</p>";

    Helpers::sendEmail($email, $subject, $body);

    $this->view('auth/forgot_password', ['success' => 'Se o email existir, um link de recuperação foi enviado.']);
  }

  public function resetPassword($token)
  {
    $testToken = filter_var($token, FILTER_SANITIZE_SPECIAL_CHARS); // Basic sanitize

    // Validate Token
    $db = new Database();
    $conn = $db->connect();

    $stmt = $conn->prepare("SELECT * FROM password_resets WHERE token = ? AND created_at > (NOW() - INTERVAL 1 HOUR)");
    $stmt->execute([$testToken]);
    $reset = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$reset) {
      $this->view('auth/forgot_password', ['error' => 'Link inválido ou expirado. Solicite novamente.']);
      return;
    }

    $this->view('auth/reset_password', ['token' => $testToken]);
  }

  public function updatePassword()
  {
    $token = $_POST['token'];
    $password = $_POST['password'];
    $confirm = $_POST['confirm_password'];

    if ($password !== $confirm) {
      $this->view('auth/reset_password', ['token' => $token, 'error' => 'Senhas não coincidem.']);
      return;
    }

    if (strlen($password) < 6) {
      $this->view('auth/reset_password', ['token' => $token, 'error' => 'Senha muito curta.']);
      return;
    }

    $db = new Database();
    $conn = $db->connect();

    // Get Email from Token
    $stmt = $conn->prepare("SELECT email FROM password_resets WHERE token = ?");
    $stmt->execute([$token]);
    $reset = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$reset) {
      $this->view('auth/forgot_password', ['error' => 'Sessão expirada.']);
      return;
    }

    $email = $reset['email'];
    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);

    // Update User
    $userModel = $this->model('User');
    $user = $userModel->findByEmail($email);
    if ($user) {
      // Manual update to be safe and quick
      $stmtUpd = $conn->prepare("UPDATE users SET password = ? WHERE id = ?");
      $stmtUpd->execute([$hashedPassword, $user['id']]);

      // Delete Token
      $conn->prepare("DELETE FROM password_resets WHERE email = ?")->execute([$email]);

      $this->view('auth/login', ['success' => 'Senha alterada com sucesso! Faça login.']);
    } else {
      $this->view('auth/forgot_password', ['error' => 'Usuário não encontrado.']);
    }
  }
}
