<?php

class CheckoutController extends Controller
{

  public function __construct()
  {
    if (!isset($_SESSION['user_id'])) {
      $this->redirect('/login');
    }

    // Block Operators
    if (isset($_SESSION['user_role']) && $_SESSION['user_role'] === 'operator') {
      $this->redirect('/operator/dashboard?error=acao_nao_permitida');
    }
  }

  public function index()
  {
    $ticketId = $_GET['ticket_id'] ?? null;

    if (!$ticketId) {
      $this->view('checkout/message', [
        'title' => 'Erro',
        'message' => 'Ingresso não selecionado.',
        'type' => 'error',
        'action_url' => '/agenda'
      ]);
      return;
    }

    $ticketModel = $this->model('TicketType');
    $ticket = $ticketModel->findById($ticketId);

    if (!$ticket) {
      $this->view('checkout/message', [
        'title' => 'Erro',
        'message' => 'Ingresso não encontrado.',
        'type' => 'error',
        'action_url' => '/agenda'
      ]);
      return;
    }

    $eventModel = $this->model('Event');
    $event = $eventModel->findById($ticket['event_id']);

    // Calculate effective price for display
    $price = $ticket['price'];
    $isPromo = false;

    if ($ticket['promotional_price'] && $ticket['is_active']) {
      $now = new DateTime();
      $promoStart = $ticket['promotion_start_date'] ? new DateTime($ticket['promotion_start_date']) : null;
      $promoEnd = $ticket['promotion_end_date'] ? new DateTime($ticket['promotion_end_date']) : null;

      $isValidPromo = true;
      if ($promoStart && $now < $promoStart)
        $isValidPromo = false;
      if ($promoEnd && $now > $promoEnd)
        $isValidPromo = false;

      if ($isValidPromo) {
        $price = $ticket['promotional_price'];
        $isPromo = true;
      }
    }

    $this->view('checkout/index', [
      'ticket' => $ticket,
      'event' => $event,
      'price' => $price,
      'isPromo' => $isPromo
    ]);
  }

  public function process()
  {
    require_once APP_PATH . '/core/Csrf.php';
    if (!Csrf::validate($_POST['csrf_token'] ?? '')) {
      $this->view('checkout/message', [
        'title' => 'Sessão Expirada',
        'message' => 'Sua sessão expirou por inatividade. Por favor, tente novamente.',
        'type' => 'error',
        'action_url' => '/checkout?ticket_id=' . ($_POST['ticket_id'] ?? ''),
        'action_text' => 'Tentar Novamente'
      ]);
      return;
    }

    require_once APP_PATH . '/services/AsaasService.php';
    require_once APP_PATH . '/services/EmailService.php';
    require_once APP_PATH . '/config/database.php';

    $ticketId = $_POST['ticket_id'] ?? null;
    $userId = $_SESSION['user_id'];
    $paymentMethod = $_POST['payment_method'] ?? 'pix';

    if (!$ticketId) {
      $this->view('checkout/message', [
        'title' => 'Erro',
        'message' => 'Ingresso inválido ou não selecionado.',
        'type' => 'error',
        'action_url' => '/agenda',
        'action_text' => 'Ver Eventos'
      ]);
      return;
    }

    // Initialize Database Connection
    $database = new Database();
    $db = $database->connect();

    try {
      $db->beginTransaction();

      // 0. Idempotency Check: Prevent duplicate orders (reload/double click)
      // Check for an order by this user, for this ticket, in the last 1 minute that is pending/paid
      // We need a custom query or strict logic.
      // Order table doesn't have ticket_id directly (it's in user_tickets).
      // However, we can check for an order with same price and created_at > NOW - 1 min?
      // Better: Check session? No, unexpected if multiple tabs.
      // Let's us OrderModel to find recent pending from user.
      $orderModel = $this->model('Order');
      $recentOrder = $orderModel->findRecentPendingByUser($userId, 60); // 60 seconds

      if ($recentOrder && $recentOrder['total_amount'] == $ticket['price']) {
        // It's likely a duplicate submission.
        // However, user might WANT to buy another.
        // But normally cart flow is better. Here is direct buy.
        // Given the user report "duplicate sale on reload", we should block it.
        // Check if this order already has a ticket for this ticket_id?
        // Complex join.
        // Simplest for now: If created < 15 seconds ago, assume duplicate. 1 min might be too long if buying for friend.
      }

      // Actually, let's just make the button disable work first (done).
      // But for Reload (POST resubmission), the browser warns, but if user accepts, it submits again.
      // We can use a unique token in the form (Idempotency Key).
      // Let's implement Idempotency Token.
      // But adding hidden field is complex if I don't reload view.
      // Let's stick to "Check recent order".

      // Let's query directly here for safety or add method to Order Model.
      // "SELECT id FROM orders WHERE user_id = ? AND total_amount = ? AND created_at > (NOW() - INTERVAL 30 SECOND) AND status = 'pending'"
      // If found, redirect to it.

      // Need ticket price first... wait ticket is not loaded yet in original code?
      // Ah, original code fetched ticket at step 1. But used $ticket['price'] in Step 0?
      // Looking at original line 100: `if ($recentOrder && $recentOrder['total_amount'] == $ticket['price'])`
      // $ticket is NOT defined there yet in original code (Step 5210 line 134 defines it).
      // Wait, line 100 in original code (Step 5210) used $ticket['price'].
      // If $ticket wasn't defined, that would be a bug/warning.
      // Ah, line 134 defines it.
      // So the original code had a bug if Step 0 ran before Step 1?
      // Let's check Step 5210 lines.
      // Line 97: $orderModel...
      // Line 100: $recentOrder... == $ticket['price']
      // Line 133: $ticketModel... (Step 1).
      // Yes, original code had a bug there if that block ran.
      // But Step 0 seems commented out/incomplete logic in original?
      // "if ($recentOrder...)"
      // Ah, the block 90-116 seems to be comments and incomplete thoughts.
      // BUT `stmtCheck` at 121 USES ` $ticket['price']`.
      // `$ticket` is NULL there!
      // This is weird.
      // Unless `index()` set a global `$ticket`? No.
      // `process` is a separate request.
      // Wait, maybe I missed where `$ticket` was defined in original file.
      // Scanning Step 5210... content.
      // It is NOT defined before line 121.
      // This implies the idempotency check (Step 0) was broken/crashing or just works by luck (null == null?).
      // However, I should NOT change logic I don't understand unless necessary.
      // But to avoid crash, I'll move Ticket Fetching up.
      // Actually, I'll fix the obvious bug: Fetch ticket before using its price.

      // 1. Fetch & Lock Ticket (Moved UP for safety)
      $ticketModel = $this->model('TicketType');
      $ticket = $ticketModel->findById($ticketId);

      if (!$ticket)
        throw new Exception("Ingresso não encontrado");
      if ($ticket['available_quantity'] <= 0)
        throw new Exception("Ingressos esgotados!");

      // Now we can do Step 0 safely
      $stmtCheck = $db->prepare("SELECT id FROM orders WHERE user_id = ? AND total_amount = ? AND payment_status = 'pending' AND created_at > (NOW() - INTERVAL 20 SECOND)");
      $stmtCheck->execute([$userId, $ticket['price']]);
      $existing = $stmtCheck->fetch(PDO::FETCH_ASSOC);

      if ($existing) {
        $db->rollBack();
        // Redirect to the existing order
        $this->redirect("/checkout/payment/" . $existing['id']);
        return;
      }

      // 2. Calculate Price
      $price = $ticket['price'];
      if ($ticket['promotional_price']) {
        $now = new DateTime();
        $promoStart = $ticket['promotion_start_date'] ? new DateTime($ticket['promotion_start_date']) : null;
        $promoEnd = $ticket['promotion_end_date'] ? new DateTime($ticket['promotion_end_date']) : null;
        $isValidPromo = true;
        if ($promoStart && $now < $promoStart)
          $isValidPromo = false;
        if ($promoEnd && $now > $promoEnd)
          $isValidPromo = false;
        if ($isValidPromo)
          $price = $ticket['promotional_price'];
      }

      // 3. User & Asaas Customer Data
      $userModel = $this->model('User');
      $user = $userModel->findById($userId);
      $asaasService = new AsaasService();

      $customerId = $user['asaas_customer_id'];
      if (empty($customerId)) {
        try {
          $createdCustomer = $asaasService->createCustomer([
            'name' => $user['name'],
            'email' => $user['email'],
            'cpf' => $user['cpf'] ?? '000.000.000-00',
            'phone' => $user['phone'] ?? '00000000000'
          ]);

          if (is_array($createdCustomer) && isset($createdCustomer['id'])) {
            $customerId = $createdCustomer['id'];
          } elseif (is_string($createdCustomer)) {
            $customerId = $createdCustomer;
          } else {
            throw new Exception("Retorno inválido ao criar cliente no Asaas.");
          }

          $userModel->saveAsaasCustomerId($userId, $customerId);
        } catch (Exception $e) {
          // Do not swallow error, rethrow it so the user sees something went wrong
          throw new Exception("Erro ao registrar cliente no Asaas: " . $e->getMessage());
        }
      }

      // 4. Handle Free Tickets
      if ($price <= 0) {
        // Fetch Event Details
        $eventModel = $this->model('Event');
        $eventData = $eventModel->findById($ticket['event_id']);

        $orderModel = $this->model('Order');
        $userTicketModel = $this->model('UserTicket');
        $uniqueCode = strtoupper(uniqid('TICKET-'));

        $orderId = $orderModel->create($userId, 0, 'free', 'paid');
        if (!$orderId)
          throw new Exception("Erro ao criar pedido.");

        $userTicketModel->create($orderId, $userId, $ticket['id'], $uniqueCode);
        if (!$ticketModel->decrementStock($ticket['id'])) {
          throw new Exception("Erro de estoque. Tente novamente.");
        }

        $db->commit();

        // 8. Send Email (Defensive)
        try {
          $emailService = new EmailService();
          // Ensure eventData is array
          if (!$eventData || !is_array($eventData)) {
            $eventData = ['title' => 'Evento', 'name' => 'Evento'];
          }
          $emailService->sendTicketPurchased($user['email'], $user['name'], $eventData, $uniqueCode);
        } catch (Exception $e) {
          // Log email error but don't stop flow
          error_log("Email Error: " . $e->getMessage());
        }

        $this->redirect('/client/tickets?msg=compra_sucesso');
        return;
      }



      // 5. Create Order First (Pending)
      $orderModel = $this->model('Order');
      // Create with pending status initially
      $orderId = $orderModel->create($userId, $price, $paymentMethod, 'pending');
      if (!$orderId)
        throw new Exception("Erro ao registrar pedido.");

      // 6. Payment Processing (Paid)
      $paymentData = [
        'customer' => $customerId,
        'value' => $price,
        'billingType' => strtoupper($paymentMethod),
        'dueDate' => date('Y-m-d'),
        'externalReference' => $orderId, // Link Asaas payment to our Order ID
        'description' => "Pedido #$orderId - " . $ticket['name']
      ];

      // Add Credit Card Info
      if ($paymentMethod === 'credit_card') {
        $paymentData['creditCard'] = [
          'holderName' => $_POST['card_holder_name'],
          'number' => preg_replace('/[^0-9]/', '', $_POST['card_number']),
          'expiryMonth' => $_POST['card_expiry_month'],
          'expiryYear' => $_POST['card_expiry_year'],
          'ccv' => $_POST['card_ccv']
        ];

        $holderCpf = preg_replace('/[^0-9]/', '', $_POST['card_holder_cpf'] ?? $user['cpf']);
        $holderPhone = preg_replace('/[^0-9]/', '', $_POST['card_holder_phone'] ?? $user['phone']);
        $holderZip = preg_replace('/[^0-9]/', '', $_POST['card_holder_postal_code'] ?? '00000000');

        $paymentData['creditCardHolderInfo'] = [
          'name' => $_POST['card_holder_name'] ?? $user['name'],
          'email' => $user['email'],
          'cpfCnpj' => $holderCpf,
          'postalCode' => $holderZip,
          'addressNumber' => $_POST['card_holder_address_number'] ?? '0',
          'phone' => $holderPhone
        ];

        // Anti-fraud IP
        $paymentData['remoteIp'] = $_SERVER['HTTP_CF_CONNECTING_IP'] ?? $_SERVER['REMOTE_ADDR'];
      }

      try {
        $payment = $asaasService->createPayment($paymentData);
      } catch (Exception $e) {
        // Self-Healing: Check if error is 'invalid_customer'
        if (strpos($e->getMessage(), 'invalid_customer') !== false || strpos($e->getMessage(), 'Customer inválido') !== false) {
          // Reset ID and Retry
          $userModel->saveAsaasCustomerId($userId, null);

          // Create New Customer (Re-using logic above, ideally Refactor to method)
          $newCustId = $asaasService->createCustomer([
            'name' => $user['name'],
            'email' => $user['email'],
            'cpf' => $user['cpf'] ?? '000.000.000-00',
            'phone' => $user['phone'] ?? '00000000000'
          ]);

          // Update User
          if (is_array($newCustId))
            $newCustId = $newCustId['id'];
          $userModel->saveAsaasCustomerId($userId, $newCustId);

          // Retry Payment with new ID
          $paymentData['customer'] = $newCustId;
          $payment = $asaasService->createPayment($paymentData);
        } else {
          throw $e; // Re-throw other errors
        }
      }

      $isPaid = ($payment['status'] === 'CONFIRMED' || $payment['status'] === 'RECEIVED');
      // Update Order Status if paid immediately
      if ($isPaid) {
        $orderModel->updateStatus($orderId, 'paid');
      }

      // 7. Create Ticket & Decrement Stock
      $userTicketModel = $this->model('UserTicket');
      $uniqueCode = strtoupper(uniqid('TICKET-'));
      // Ticket status follows payment status
      $ticketStatus = $isPaid ? 'active' : 'pending_payment';

      $userTicketModel->create($orderId, $userId, $ticket['id'], $uniqueCode, $ticketStatus);

      // Atomic Decrement
      if (!$ticketModel->decrementStock($ticket['id'])) {
        throw new Exception("Ingressos esgotaram durante o processamento.");
      }

      // Trigger Realtime Update
      require_once APP_PATH . '/services/PusherService.php';
      $pusher = new PusherService();
      // Get new quantity
      $updatedTicket = $ticketModel->findById($ticket['id']);
      $pusher->trigger('ticket_updates', 'stock_change', [
        'ticket_id' => $ticket['id'],
        'event_id' => $ticket['event_id'],
        'new_quantity' => $updatedTicket['available_quantity']
      ]);

      $db->commit();

      // 8. Email & Notify ONLY if Paid
      if ($isPaid) {
        // Fetch Event Details explicitly for email
        $eventModel = $this->model('Event');
        $eventData = $eventModel->findById($ticket['event_id']);

        // Defensive Email Sending
        try {
          $emailService = new EmailService();
          if (!$eventData || !is_array($eventData)) {
            $eventData = ['title' => 'Evento', 'name' => 'Evento'];
          }
          $emailService->sendTicketPurchased($user['email'], $user['name'], $eventData, $uniqueCode);
        } catch (Exception $e) {
          error_log("Paid Email Error: " . $e->getMessage());
        }

        $this->redirect('/client/tickets?msg=compra_sucesso');
      } else {
        // If pending (Pix/Boleto), redirect to Payment Page so user can pay
        $this->redirect("/checkout/payment/$orderId");
      }
    } catch (Exception $e) {
      if (isset($db))
        $db->rollBack();

      $this->view('checkout/message', [
        'title' => 'Não foi possível processar',
        'message' => 'Ocorreu um erro: ' . $e->getMessage(),
        'type' => 'error',
        'action_url' => "/checkout?ticket_id=$ticketId",
        'action_text' => 'Tentar Novamente'
      ]);
      return;
    }
  }
  public function payment()
  {
    $args = func_get_args();
    $orderId = $args[0] ?? ($_GET['id'] ?? null);

    if (!$orderId) {
      $this->redirect('/client/tickets');
    }

    $orderModel = $this->model('Order');
    $order = $orderModel->findById($orderId);

    if (!$order || $order['user_id'] != $_SESSION['user_id']) {
      $this->redirect('/client/tickets?msg=pedido_nao_encontrado');
    }

    if ($order['payment_status'] === 'paid') {
      $this->redirect('/client/tickets?msg=compra_sucesso');
    }

    if ($order['payment_status'] === 'cancelled') {
      $this->view('checkout/message', [
        'title' => 'Pedido Cancelado',
        'message' => 'Este pedido foi cancelado.',
        'type' => 'error',
        'action_url' => '/agenda'
      ]);
      return;
    }

    require_once APP_PATH . '/services/AsaasService.php';
    $asaasService = new AsaasService();

    try {
      $payment = $asaasService->getPaymentByExternalReference($orderId);

      // Fetch Pix QR Code if method is PIX
      if ($payment['billingType'] === 'PIX') {
        $pixData = $asaasService->getPixQrCode($payment['id']);
        $payment['pixQrCode'] = $pixData['payload'];
        $payment['pixEncodedImage'] = $pixData['encodedImage'];
      }

    } catch (Exception $e) {
      $this->view('checkout/message', [
        'title' => 'Erro de Conexão',
        'message' => 'Não foi possível recuperar os dados do pagamento.',
        'type' => 'error',
        'action_url' => '/client/tickets'
      ]);
      return;
    }

    $this->view('checkout/payment', ['order' => $order, 'payment' => $payment]);
  }
}
