<?php

class OperatorController extends Controller
{
  public function __construct()
  {
    if (!isset($_SESSION['user_role']) || $_SESSION['user_role'] !== 'operator') {
      $this->redirect('/login');
    }
  }

  public function dashboard()
  {
    $userId = $_SESSION['user_id'];
    $eventModel = $this->model('Event');

    // Get events assigned to this operator
    $events = $eventModel->getEventsByOperator($userId);

    $this->view('operator/dashboard', ['events' => $events]);
  }

  public function checkin($eventId)
  {
    // Verify assignment
    $userId = $_SESSION['user_id'];
    $eventModel = $this->model('Event');

    if (!$eventModel->isOperator($eventId, $userId)) {
      die("Acesso negado: Você não é operador deste evento.");
    }

    $event = $eventModel->findById($eventId);

    // Get Check-in Count
    $userTicketModel = $this->model('UserTicket');
    $checkinCount = $userTicketModel->countByEventAndStatus($eventId, 'used');

    $this->view('operator/checkin', ['event' => $event, 'total_checkins' => $checkinCount]);
  }

  public function processCheckin()
  {
    require_once APP_PATH . '/core/Csrf.php';
    if (!Csrf::validate($_POST['csrf_token'] ?? '')) {
      die('CSRF Validation Failed');
    }

    $code = $_POST['code'] ?? '';
    // Event ID is passed to verify context?
    // Actually, ticket code is unique globally usually, but good to check if ticket belongs to event?
    // UserTicket doesn't store event_id directly (via ticket_type).
    // Let's validate code first using UserTicket::findByCode

    $userTicketModel = $this->model('UserTicket');
    $ticket = $userTicketModel->findByCode($code);

    // We should verify if the ticket belongs to an event the operator is assigned to?
    // Yes, crucial for security.
    $userId = $_SESSION['user_id'];
    $eventModel = $this->model('Event');

    if ($ticket) {
      // Check if operator is assigned to THIS ticket's event
      // Ticket -> TicketType -> Event
      // Helpers to get event id from ticket data? findByCode joins event data.
      // $ticket has 'event_title', wait, does it have event_id?
      // findByCode selects `e.*` or check SQL?
      // SQL: `e.title as event_title, e.start_date, e.location, e.image_cover` ... doesn't select e.id explicitly?
      // SQL JOINs `events e` on `tt.event_id = e.id`. 
      // I should ensure findByCode returns event_id.

      // Let's check UserTicket::findByCode SQL again.
      // It selects `ut.*`, so `ut` columns. `ticket_types` has `event_id`.
      // I should probably fetch the TicketType to get event_id, OR rely on `findByCode` result if I update it.
      // Let's trust `ticket_type_id` is in properties.

      $ticketTypeId = $ticket['ticket_type_id'];
      $ticketTypeModel = $this->model('TicketType');
      $ticketType = $ticketTypeModel->findById($ticketTypeId);
      $eventId = $ticketType['event_id'];

      if (!$eventModel->isOperator($eventId, $userId)) {
        $this->view('operator/checkin', ['error' => 'Este ingresso pertence a um evento que você não gerencia.']);
        return;
      }

      // Proceed with validation logic (same as Admin)
      if ($ticket['status'] === 'used') {
        $this->view('operator/checkin', ['error' => 'Ingresso JÁ UTILIZADO em: ' . date('d/m/Y H:i', strtotime($ticket['used_at'])), 'event' => ['id' => $eventId, 'title' => $ticket['event_title']]]);
      } elseif ($ticket['status'] === 'cancelled') {
        $this->view('operator/checkin', ['error' => 'Ingresso CANCELADO. Acesso negado.', 'event' => ['id' => $eventId, 'title' => $ticket['event_title']]]);
      } elseif ($ticket['status'] === 'pending' || $ticket['status'] === 'pending_payment') {
        $this->view('operator/checkin', ['error' => 'Pagamento PENDENTE. Não liberar entrada.', 'event' => ['id' => $eventId, 'title' => $ticket['event_title']]]);
      } elseif ($ticket['status'] === 'active') {
        $userTicketModel->markAsUsed($ticket['id'], $userId);

        // Refresh count
        $newCount = $userTicketModel->countByEventAndStatus($eventId, 'used');

        $this->view('operator/checkin', ['success' => true, 'ticket' => $ticket, 'event' => ['id' => $eventId, 'title' => $ticket['event_title']], 'total_checkins' => $newCount]);
      } else {
        // Refresh count even on unknown status error
        $currentCount = $userTicketModel->countByEventAndStatus($eventId, 'used');
        $this->view('operator/checkin', ['error' => 'Status desconhecido.', 'event' => ['id' => $eventId, 'title' => $ticket['event_title']], 'total_checkins' => $currentCount]);
      }

    } else {
      // Need event context to re-render view properly if we want 'Back' button to work?
      // Or just generic error.
      // Ideally pass event_id in form?
      // For now, simple error.
      // But wait, without event object, the view might break if it expects $event?
      // I should grab event from POST if available to re-render view context.
      // Or better, just show error.
      $this->view('operator/checkin', ['error' => 'Código inválido ou não encontrado.']);
    }
  }
}
