<?php

class Event extends Model
{

  public function count()
  {
    $stmt = $this->conn->query("SELECT COUNT(*) FROM events");
    return $stmt->fetchColumn();
  }

  public function getAllOthers($excludeId)
  {
    $stmt = $this->conn->prepare("SELECT * FROM events WHERE status = 'published' AND id != :id ORDER BY start_date ASC");
    $stmt->bindParam(':id', $excludeId);
    $stmt->execute();
    return $stmt->fetchAll();
  }

  public function getAllPublished()
  {
    $stmt = $this->conn->prepare("SELECT * FROM events WHERE status = 'published' ORDER BY start_date ASC");
    $stmt->execute();
    return $stmt->fetchAll();
  }

  public function getAll()
  {
    $stmt = $this->conn->prepare("SELECT * FROM events ORDER BY start_date DESC");
    $stmt->execute();
    return $stmt->fetchAll();
  }

  public function findById($id)
  {
    $stmt = $this->conn->prepare("SELECT * FROM events WHERE id = :id");
    $stmt->bindParam(':id', $id);
    $stmt->execute();
    return $stmt->fetch();
  }

  public function findBySlug($slug)
  {
    $stmt = $this->conn->prepare("SELECT * FROM events WHERE slug = :slug");
    $stmt->bindParam(':slug', $slug);
    $stmt->execute();
    return $stmt->fetch();
  }

  public function create($data)
  {
    $sql = "INSERT INTO events (title, slug, description, location, start_date, image_cover, status, organizer_name, organizer_phone, map_embed, marketing_content, social_instagram, social_linkedin, social_website, social_whatsapp) VALUES (:title, :slug, :description, :location, :start_date, :image_cover, :status, :organizer_name, :organizer_phone, :map_embed, :marketing_content, :social_instagram, :social_linkedin, :social_website, :social_whatsapp)";

    $stmt = $this->conn->prepare($sql);

    $stmt->bindValue(':title', $data['title']);
    $stmt->bindValue(':slug', $data['slug']);
    $stmt->bindValue(':description', $data['description']);
    $stmt->bindValue(':location', $data['location']);
    $stmt->bindValue(':start_date', $data['start_date']);
    $stmt->bindValue(':image_cover', $data['image_cover']);
    $stmt->bindValue(':status', $data['status'] ?? 'draft');

    $stmt->bindValue(':organizer_name', $data['organizer_name'] ?? null);
    $stmt->bindValue(':organizer_phone', $data['organizer_phone'] ?? null);
    $stmt->bindValue(':map_embed', $data['map_embed'] ?? null);
    $stmt->bindValue(':marketing_content', $data['marketing_content'] ?? null);

    $stmt->bindValue(':social_instagram', $data['social_instagram'] ?? null);
    $stmt->bindValue(':social_linkedin', $data['social_linkedin'] ?? null);
    $stmt->bindValue(':social_website', $data['social_website'] ?? null);
    $stmt->bindValue(':social_whatsapp', $data['social_whatsapp'] ?? null);

    if ($stmt->execute()) {
      return $this->conn->lastInsertId();
    }
    return false;
  }

  public function update($id, $data)
  {
    $sql = "UPDATE events SET 
              title = :title, 
              slug = :slug, 
              description = :description, 
              location = :location, 
              start_date = :start_date, 
              image_cover = :image_cover, 
              status = :status,
              organizer_name = :organizer_name,
              organizer_phone = :organizer_phone,
              map_embed = :map_embed,
              marketing_content = :marketing_content,
              social_instagram = :social_instagram,
              social_linkedin = :social_linkedin,
              social_website = :social_website,
              social_whatsapp = :social_whatsapp
            WHERE id = :id";

    $stmt = $this->conn->prepare($sql);

    $stmt->bindValue(':title', $data['title']);
    $stmt->bindValue(':slug', $data['slug']);
    $stmt->bindValue(':description', $data['description']);
    $stmt->bindValue(':location', $data['location']);
    $stmt->bindValue(':start_date', $data['start_date']);
    $stmt->bindValue(':image_cover', $data['image_cover']);
    $stmt->bindValue(':status', $data['status']);

    // Optional fields with defaults
    $stmt->bindValue(':organizer_name', $data['organizer_name'] ?? null);
    $stmt->bindValue(':organizer_phone', $data['organizer_phone'] ?? null);
    $stmt->bindValue(':map_embed', $data['map_embed'] ?? null);
    $stmt->bindValue(':marketing_content', $data['marketing_content'] ?? null);

    $stmt->bindValue(':social_instagram', $data['social_instagram'] ?? null);
    $stmt->bindValue(':social_linkedin', $data['social_linkedin'] ?? null);
    $stmt->bindValue(':social_website', $data['social_website'] ?? null);
    $stmt->bindValue(':social_whatsapp', $data['social_whatsapp'] ?? null);

    $stmt->bindValue(':id', $id);

    return $stmt->execute();
  }

  public function getHighlighted()
  {
    $stmt = $this->conn->prepare("SELECT * FROM events WHERE is_highlight = 1 AND status = 'published' LIMIT 1");
    $stmt->execute();
    return $stmt->fetch();
  }

  public function clearHighlights()
  {
    $stmt = $this->conn->prepare("UPDATE events SET is_highlight = 0");
    return $stmt->execute();
  }

  public function markAsHighlight($id)
  {
    $stmt = $this->conn->prepare("UPDATE events SET is_highlight = 1 WHERE id = :id");
    $stmt->bindParam(':id', $id);
    $stmt->bindParam(':id', $id);
    return $stmt->execute();
  }



  public function delete($id)
  {
    $stmt = $this->conn->prepare("DELETE FROM events WHERE id = :id");
    $stmt->bindParam(':id', $id);
    return $stmt->execute();
  }

  // --- Operator Management ---

  public function addOperator($eventId, $userId)
  {
    // Ignore if already exists (INSERT IGNORE or try-catch)
    $sql = "INSERT INTO event_operators (event_id, user_id) VALUES (:event_id, :user_id)";
    try {
      $stmt = $this->conn->prepare($sql);
      $stmt->bindValue(':event_id', $eventId);
      $stmt->bindValue(':user_id', $userId);
      return $stmt->execute();
    } catch (PDOException $e) {
      if ($e->getCode() == 23000)
        return true; // Duplicate entry
      return false;
    }
  }

  public function removeOperator($eventId, $userId)
  {
    $stmt = $this->conn->prepare("DELETE FROM event_operators WHERE event_id = :event_id AND user_id = :user_id");
    $stmt->bindValue(':event_id', $eventId);
    $stmt->bindValue(':user_id', $userId);
    return $stmt->execute();
  }

  public function getOperators($eventId)
  {
    $sql = "SELECT u.id, u.name, u.email 
            FROM users u
            JOIN event_operators eo ON u.id = eo.user_id
            WHERE eo.event_id = :event_id
            ORDER BY u.name ASC";
    $stmt = $this->conn->prepare($sql);
    $stmt->bindValue(':event_id', $eventId);
    $stmt->execute();
    return $stmt->fetchAll();
  }

  public function getEventsByOperator($userId)
  {
    $sql = "SELECT e.* 
            FROM events e
            JOIN event_operators eo ON e.id = eo.event_id
            WHERE eo.user_id = :user_id AND e.status = 'published'
            ORDER BY e.start_date ASC";
    $stmt = $this->conn->prepare($sql);
    $stmt->bindValue(':user_id', $userId);
    $stmt->execute();
    return $stmt->fetchAll();
  }

  public function isOperator($eventId, $userId)
  {
    $stmt = $this->conn->prepare("SELECT COUNT(*) FROM event_operators WHERE event_id = :event_id AND user_id = :user_id");
    $stmt->bindValue(':event_id', $eventId);
    $stmt->bindValue(':user_id', $userId);
    $stmt->execute();
    return $stmt->fetchColumn() > 0;
  }
}
