<?php

class Order extends Model
{
  public function create($userId, $totalAmount, $paymentMethod, $paymentStatus = 'pending')
  {
    $sql = "INSERT INTO orders (user_id, total_amount, payment_method, payment_status, created_at) VALUES (:user_id, :total_amount, :payment_method, :payment_status, NOW())";
    $stmt = $this->conn->prepare($sql);
    $stmt->bindValue(':user_id', $userId);
    $stmt->bindValue(':total_amount', $totalAmount);
    $stmt->bindValue(':payment_method', $paymentMethod);
    $stmt->bindValue(':payment_status', $paymentStatus);

    if ($stmt->execute()) {
      return $this->conn->lastInsertId();
    }
    return false;
  }

  public function countByUserId($userId)
  {
    $stmt = $this->conn->prepare("SELECT COUNT(*) FROM orders WHERE user_id = :user_id");
    $stmt->bindValue(':user_id', $userId);
    $stmt->execute();
    return $stmt->fetchColumn();
  }

  public function updateStatus($id, $status)
  {
    $sql = "UPDATE orders SET payment_status = :status WHERE id = :id";
    $stmt = $this->conn->prepare($sql);
    $stmt->bindValue(':status', $status);
    $stmt->bindValue(':id', $id);
    return $stmt->execute();
  }

  public function findById($id)
  {
    $stmt = $this->conn->prepare("SELECT * FROM orders WHERE id = :id");
    $stmt->bindParam(':id', $id);
    $stmt->execute();
    return $stmt->fetch();
  }

  public function findRecentPendingByUser($userId, $seconds = 60)
  {
    $sql = "SELECT * FROM orders 
            WHERE user_id = :user_id 
            AND payment_status = 'pending' 
            AND created_at > (NOW() - INTERVAL :seconds SECOND) 
            ORDER BY created_at DESC LIMIT 1";

    $stmt = $this->conn->prepare($sql);
    $stmt->bindValue(':user_id', $userId);
    $stmt->bindValue(':seconds', (int) $seconds, PDO::PARAM_INT);
    $stmt->execute();
    return $stmt->fetch();
  }

  public function count($filters = [])
  {
    $sql = "SELECT COUNT(DISTINCT o.id) FROM orders o
            JOIN user_tickets ut ON o.id = ut.order_id
            JOIN ticket_types tt ON ut.ticket_type_id = tt.id
            WHERE 1=1";

    if (!empty($filters['event_id'])) {
      $sql .= " AND tt.event_id = :event_id";
    }
    if (!empty($filters['ticket_id'])) {
      $sql .= " AND tt.id = :ticket_id";
    }

    $stmt = $this->conn->prepare($sql);
    if (!empty($filters['event_id']))
      $stmt->bindValue(':event_id', $filters['event_id']);
    if (!empty($filters['ticket_id']))
      $stmt->bindValue(':ticket_id', $filters['ticket_id']);

    $stmt->execute();
    return $stmt->fetchColumn();
  }

  public function sumTotal($filters = [])
  {
    $sql = "SELECT SUM(o.total_amount) FROM orders o
            JOIN user_tickets ut ON o.id = ut.order_id
            JOIN ticket_types tt ON ut.ticket_type_id = tt.id
            WHERE o.payment_status = 'paid'";

    if (!empty($filters['event_id'])) {
      $sql .= " AND tt.event_id = :event_id";
    }
    if (!empty($filters['ticket_id'])) {
      $sql .= " AND tt.id = :ticket_id";
    }

    $stmt = $this->conn->prepare($sql);
    if (!empty($filters['event_id']))
      $stmt->bindValue(':event_id', $filters['event_id']);
    if (!empty($filters['ticket_id']))
      $stmt->bindValue(':ticket_id', $filters['ticket_id']);

    $stmt->execute();
    return $stmt->fetchColumn() ?: 0;
  }

  public function getDailySales($filters = [])
  {
    $sql = "SELECT DATE(o.created_at) as date, COUNT(DISTINCT o.id) as count, SUM(o.total_amount) as revenue
            FROM orders o
            JOIN user_tickets ut ON o.id = ut.order_id
            JOIN ticket_types tt ON ut.ticket_type_id = tt.id
            WHERE o.payment_status = 'paid'";

    if (!empty($filters['event_id'])) {
      $sql .= " AND tt.event_id = :event_id";
    }
    if (!empty($filters['ticket_id'])) {
      $sql .= " AND tt.id = :ticket_id";
    }
    if (!empty($filters['start_date'])) {
      $sql .= " AND o.created_at >= :start_date";
    }
    if (!empty($filters['end_date'])) {
      $sql .= " AND o.created_at <= :end_date";
    }

    $sql .= " GROUP BY DATE(o.created_at) ORDER BY date ASC LIMIT 30";

    $stmt = $this->conn->prepare($sql);
    if (!empty($filters['event_id']))
      $stmt->bindValue(':event_id', $filters['event_id']);
    if (!empty($filters['ticket_id']))
      $stmt->bindValue(':ticket_id', $filters['ticket_id']);
    if (!empty($filters['start_date']))
      $stmt->bindValue(':start_date', $filters['start_date'] . ' 00:00:00');
    if (!empty($filters['end_date']))
      $stmt->bindValue(':end_date', $filters['end_date'] . ' 23:59:59');

    $stmt->execute();
    return $stmt->fetchAll();
  }

  public function getRecentOrders($limit = 10, $filters = [])
  {
    $sql = "SELECT o.*, u.name as user_name, e.title as event_title
              FROM orders o
              LEFT JOIN users u ON o.user_id = u.id
              JOIN user_tickets ut ON o.id = ut.order_id
              JOIN ticket_types tt ON ut.ticket_type_id = tt.id
              JOIN events e ON tt.event_id = e.id
              WHERE 1=1";

    if (!empty($filters['event_id'])) {
      $sql .= " AND tt.event_id = :event_id";
    }
    if (!empty($filters['ticket_id'])) {
      $sql .= " AND tt.id = :ticket_id";
    }

    $sql .= " GROUP BY o.id, u.name, e.title ORDER BY o.created_at DESC LIMIT " . (int) $limit;

    $stmt = $this->conn->prepare($sql);
    if (!empty($filters['event_id']))
      $stmt->bindValue(':event_id', $filters['event_id']);
    if (!empty($filters['ticket_id']))
      $stmt->bindValue(':ticket_id', $filters['ticket_id']);

    $stmt->execute();
    return $stmt->fetchAll();
  }

  public function getSalesByTicketType($filters = [])
  {
    $sql = "SELECT tt.name, COUNT(ut.id) as count, SUM(o.total_amount) as revenue
              FROM ticket_types tt
              LEFT JOIN user_tickets ut ON tt.id = ut.ticket_type_id
              LEFT JOIN orders o ON ut.order_id = o.id AND o.payment_status = 'paid'
              WHERE 1=1";

    if (!empty($filters['event_id'])) {
      $sql .= " AND tt.event_id = :event_id";
    }
    if (!empty($filters['ticket_id'])) {
      $sql .= " AND tt.id = :ticket_id";
    }
    if (!empty($filters['start_date'])) {
      $sql .= " AND o.created_at >= :start_date";
    }
    if (!empty($filters['end_date'])) {
      $sql .= " AND o.created_at <= :end_date";
    }

    $sql .= " GROUP BY tt.id, tt.name";

    $stmt = $this->conn->prepare($sql);
    if (!empty($filters['event_id']))
      $stmt->bindValue(':event_id', $filters['event_id']);
    if (!empty($filters['ticket_id']))
      $stmt->bindValue(':ticket_id', $filters['ticket_id']);
    if (!empty($filters['start_date']))
      $stmt->bindValue(':start_date', $filters['start_date'] . ' 00:00:00');
    if (!empty($filters['end_date']))
      $stmt->bindValue(':end_date', $filters['end_date'] . ' 23:59:59');

    $stmt->execute();
    return $stmt->fetchAll();
  }

  public function getSalesByEvent($filters = [])
  {
    $sql = "SELECT e.id, e.title, COUNT(DISTINCT o.id) as total_orders, SUM(o.total_amount) as total_revenue
              FROM orders o
              JOIN user_tickets ut ON o.id = ut.order_id
              JOIN ticket_types tt ON ut.ticket_type_id = tt.id
              JOIN events e ON tt.event_id = e.id
              WHERE o.payment_status = 'paid'";

    if (!empty($filters['event_id'])) {
      $sql .= " AND tt.event_id = :event_id";
    }
    if (!empty($filters['ticket_id'])) {
      $sql .= " AND tt.id = :ticket_id";
    }
    if (!empty($filters['start_date'])) {
      $sql .= " AND o.created_at >= :start_date";
    }
    if (!empty($filters['end_date'])) {
      $sql .= " AND o.created_at <= :end_date";
    }

    $sql .= " GROUP BY e.id, e.title ORDER BY total_revenue DESC";

    $stmt = $this->conn->prepare($sql);
    if (!empty($filters['event_id']))
      $stmt->bindValue(':event_id', $filters['event_id']);
    if (!empty($filters['ticket_id']))
      $stmt->bindValue(':ticket_id', $filters['ticket_id']);
    if (!empty($filters['start_date']))
      $stmt->bindValue(':start_date', $filters['start_date'] . ' 00:00:00');
    if (!empty($filters['end_date']))
      $stmt->bindValue(':end_date', $filters['end_date'] . ' 23:59:59');

    $stmt->execute();
    return $stmt->fetchAll();
  }

  public function getItemsByOrder($orderId)
  {
    $sql = "SELECT ut.*, tt.name as ticket_name, tt.price as price_paid, e.title as event_title
            FROM user_tickets ut
            JOIN ticket_types tt ON ut.ticket_type_id = tt.id
            JOIN events e ON tt.event_id = e.id
            WHERE ut.order_id = :order_id";
    $stmt = $this->conn->prepare($sql);
    $stmt->bindValue(':order_id', $orderId);
    $stmt->execute();
    return $stmt->fetchAll();
  }
  public function cancelExpiredPending($minutes = 10)
  {
    // 1. Find Expired Orders
    // Using simple approach: Update directly and check row count, or select then update.
    // To update tickets, we need the Order IDs.

    // Select IDs first
    $sql = "SELECT id FROM orders 
            WHERE payment_status = 'pending' 
            AND created_at < (NOW() - INTERVAL :minutes MINUTE)";

    $stmt = $this->conn->prepare($sql);
    $stmt->bindValue(':minutes', (int) $minutes, PDO::PARAM_INT);
    $stmt->execute();
    $expiredOrders = $stmt->fetchAll(PDO::FETCH_COLUMN);

    if (!empty($expiredOrders)) {
      // Debug Log
      $log = dirname(__DIR__, 2) . '/debug_cancel.txt';
      $checkTime = $this->conn->query("SELECT NOW()")->fetchColumn();
      foreach ($expiredOrders as $oid) {
        $o = $this->findById($oid);
        $createdAt = $o['created_at'];
        file_put_contents($log, "Cancelling Order #$oid. Created: $createdAt. Current DB Time: $checkTime. Limit: $minutes min.\n", FILE_APPEND);
      }
    }

    if (empty($expiredOrders)) {
      return 0;
    }

    $placeholders = implode(',', array_fill(0, count($expiredOrders), '?'));

    // 2. Update Orders
    $updateOrders = "UPDATE orders SET payment_status = 'cancelled' WHERE id IN ($placeholders)";
    $stmtOrders = $this->conn->prepare($updateOrders);
    $stmtOrders->execute($expiredOrders);

    // 3. Update User Tickets
    $updateTickets = "UPDATE user_tickets SET status = 'cancelled' WHERE order_id IN ($placeholders)";
    $stmtTickets = $this->conn->prepare($updateTickets);
    $stmtTickets->execute($expiredOrders);

    return count($expiredOrders);
  }
}
