<?php

class AsaasService
{
  private $apiKey;
  private $apiUrl;

  public function __construct()
  {
    require_once APP_PATH . '/models/Setting.php';
    $setting = new Setting();

    // Fetch from DB, fallback to Constant
    $this->apiKey = $setting->get('asaas_api_key', defined('ASAAS_API_KEY') ? ASAAS_API_KEY : '');
    $this->apiUrl = $setting->get('asaas_url', defined('ASAAS_URL') ? ASAAS_URL : 'https://sandbox.asaas.com/api/v3');
  }

  private function request($method, $endpoint, $data = [])
  {
    $url = $this->apiUrl . $endpoint;
    $ch = curl_init();

    $headers = [
      'Content-Type: application/json',
      'access_token: ' . $this->apiKey,
      'User-Agent: CeVai/1.0'
    ];

    $options = [
      CURLOPT_URL => $url,
      CURLOPT_RETURNTRANSFER => true,
      CURLOPT_HTTPHEADER => $headers,
      CURLOPT_CUSTOMREQUEST => $method,
      CURLOPT_SSL_VERIFYPEER => false // Allow self-signed in dev if needed, ideally true for prod
    ];

    if (!empty($data) && $method !== 'GET') {
      $options[CURLOPT_POSTFIELDS] = json_encode($data);
    }

    curl_setopt_array($ch, $options);
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);

    if ($error) {
      error_log("Asaas cURL Error: $error");
      throw new Exception("Erro de conexão com Asaas.");
    }

    $decoded = json_decode($response, true);

    if ($httpCode >= 400) {
      // Log generic error, AVOID logging full payload if card data present
      $msg = $decoded['errors'][0]['description'] ?? 'Erro na requisição Asaas';
      error_log("Asaas API Error ($httpCode): $msg");
      throw new Exception($msg);
    }

    return $decoded;
  }

  /**
   * Create or Retrieve Customer
   */
  public function createCustomer($userData)
  {
    // 1. Search existing by email
    $email = urlencode($userData['email']);
    $search = $this->request('GET', "/customers?email=$email");

    if (!empty($search['data'])) {
      return $search['data'][0]['id'];
    }

    // 2. Create New
    $payload = [
      'name' => $userData['name'],
      'email' => $userData['email'],
      'cpfCnpj' => preg_replace('/[^0-9]/', '', $userData['cpf']),
      'mobilePhone' => preg_replace('/[^0-9]/', '', $userData['phone'])
    ];

    $response = $this->request('POST', '/customers', $payload);
    return $response['id'];
  }

  /**
   * Create Payment
   */
  public function createPayment($paymentData)
  {
    return $this->request('POST', '/payments', $paymentData);
  }

  /**
   * Get Payment by Order ID
   */
  public function getPaymentByExternalReference($orderId)
  {
    $response = $this->request('GET', "/payments?externalReference=$orderId");
    if (empty($response['data'])) {
      throw new Exception("Pagamento não encontrado.");
    }
    return $response['data'][0];
  }

  /**
   * Get Pix QR Code
   */
  public function getPixQrCode($paymentId)
  {
    return $this->request('GET', "/payments/$paymentId/pixQrCode");
  }
}
